from fastapi import APIRouter, Depends, HTTPException, status
from sqlalchemy.orm import Session
from typing import List
from decimal import Decimal
from app.core import deps
from app.models import Basket, BasketItem, User, Product, ProductVariant, Child, Order, OrderItem, OrderStatusEnum
from app.schemas import basket as basket_schemas

router = APIRouter()


@router.get("/", response_model=List[basket_schemas.BasketSummary])
def list_baskets(
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_parent)
):
    """List all baskets for current parent"""
    baskets = db.query(Basket).filter(Basket.parent_id == current_user.id).all()
    
    result = []
    for basket in baskets:
        total = sum(
            (item.variant.trainer_price if item.variant.trainer_price else item.variant.admin_price) * item.quantity
            for item in basket.items
        )
        result.append({
            "id": basket.id,
            "child_id": basket.child_id,
            "club_id": basket.club_id,
            "item_count": len(basket.items),
            "total_amount": total
        })
    
    return result


@router.get("/{basket_id}", response_model=basket_schemas.BasketRead)
def get_basket(
    basket_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_parent)
):
    """Get a specific basket with all items"""
    basket = db.query(Basket).filter(
        Basket.id == basket_id,
        Basket.parent_id == current_user.id
    ).first()
    
    if not basket:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Basket not found")
    
    # Build response with computed fields
    items = []
    total = Decimal(0)
    
    for item in basket.items:
        unit_price = item.variant.trainer_price if item.variant.trainer_price else item.variant.admin_price
        total_price = unit_price * item.quantity
        total += total_price
        
        items.append({
            "id": item.id,
            "basket_id": item.basket_id,
            "product_id": item.product_id,
            "variant_id": item.variant_id,
            "quantity": item.quantity,
            "customization_text": item.customization_text,
            "product_name": item.product.name,
            "variant_size": item.variant.size,
            "unit_price": unit_price,
            "total_price": total_price
        })
    
    return {
        "id": basket.id,
        "parent_id": basket.parent_id,
        "child_id": basket.child_id,
        "club_id": basket.club_id,
        "created_at": basket.created_at,
        "updated_at": basket.updated_at,
        "items": items,
        "total_amount": total
    }


@router.post("/", response_model=basket_schemas.BasketRead, status_code=status.HTTP_201_CREATED)
def create_basket(
    basket_in: basket_schemas.BasketCreate,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_parent)
):
    """Create a new basket for a child"""
    # Verify child exists and belongs to parent
    child = db.query(Child).filter(
        Child.id == basket_in.child_id,
        Child.parent_id == current_user.id
    ).first()
    
    if not child:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail="Child not found or doesn't belong to you"
        )
    
    # Check if basket already exists for this child and club
    existing_basket = db.query(Basket).filter(
        Basket.parent_id == current_user.id,
        Basket.child_id == basket_in.child_id,
        Basket.club_id == basket_in.club_id
    ).first()
    
    if existing_basket:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Basket already exists for this child and club"
        )
    
    # Create basket
    basket = Basket(
        parent_id=current_user.id,
        child_id=basket_in.child_id,
        club_id=basket_in.club_id
    )
    db.add(basket)
    db.commit()
    db.refresh(basket)
    
    return {
        "id": basket.id,
        "parent_id": basket.parent_id,
        "child_id": basket.child_id,
        "club_id": basket.club_id,
        "created_at": basket.created_at,
        "updated_at": basket.updated_at,
        "items": [],
        "total_amount": Decimal(0)
    }


@router.post("/{basket_id}/items", status_code=status.HTTP_201_CREATED)
def add_item_to_basket(
    basket_id: int,
    item_in: basket_schemas.BasketItemCreate,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_parent)
):
    """Add an item to a basket"""
    # Verify basket belongs to parent
    basket = db.query(Basket).filter(
        Basket.id == basket_id,
        Basket.parent_id == current_user.id
    ).first()
    
    if not basket:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Basket not found")
    
    # Verify product exists and is confirmed
    product = db.query(Product).filter(Product.id == item_in.product_id).first()
    if not product:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Product not found")
    
    if not product.is_confirmed:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Product is not available yet"
        )
    
    # Verify variant exists and belongs to product
    variant = db.query(ProductVariant).filter(
        ProductVariant.id == item_in.variant_id,
        ProductVariant.product_id == item_in.product_id
    ).first()
    
    if not variant:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Variant not found")
    
    customization_text = item_in.customization_text.strip() if item_in.customization_text else None

    # Check if item already exists in basket (match customization text)
    existing_item = db.query(BasketItem).filter(
        BasketItem.basket_id == basket_id,
        BasketItem.variant_id == item_in.variant_id,
        BasketItem.customization_text == customization_text
    ).first()
    
    if existing_item:
        # Update quantity
        existing_item.quantity += item_in.quantity
        db.commit()
        return {"message": "Item quantity updated", "item_id": existing_item.id}
    
    # Create new basket item
    basket_item = BasketItem(
        basket_id=basket_id,
        product_id=item_in.product_id,
        variant_id=item_in.variant_id,
        quantity=item_in.quantity,
        customization_text=customization_text
    )
    db.add(basket_item)
    db.commit()
    db.refresh(basket_item)
    
    return {"message": "Item added to basket", "item_id": basket_item.id}


@router.put("/{basket_id}/items/{item_id}")
def update_basket_item(
    basket_id: int,
    item_id: int,
    item_update: basket_schemas.BasketItemUpdate,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_parent)
):
    """Update item quantity in basket"""
    # Verify basket belongs to parent
    basket = db.query(Basket).filter(
        Basket.id == basket_id,
        Basket.parent_id == current_user.id
    ).first()
    
    if not basket:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Basket not found")
    
    # Find item
    item = db.query(BasketItem).filter(
        BasketItem.id == item_id,
        BasketItem.basket_id == basket_id
    ).first()
    
    if not item:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Item not found in basket")
    
    # Update quantity
    item.quantity = item_update.quantity
    db.commit()
    
    return {"message": "Item updated"}


@router.delete("/{basket_id}/items/{item_id}", status_code=status.HTTP_204_NO_CONTENT)
def remove_item_from_basket(
    basket_id: int,
    item_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_parent)
):
    """Remove an item from basket"""
    # Verify basket belongs to parent
    basket = db.query(Basket).filter(
        Basket.id == basket_id,
        Basket.parent_id == current_user.id
    ).first()
    
    if not basket:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Basket not found")
    
    # Find and delete item
    item = db.query(BasketItem).filter(
        BasketItem.id == item_id,
        BasketItem.basket_id == basket_id
    ).first()
    
    if not item:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Item not found in basket")
    
    db.delete(item)
    db.commit()
    
    return None


@router.post("/{basket_id}/checkout", status_code=status.HTTP_201_CREATED)
def checkout_basket(
    basket_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_parent)
):
    """Convert basket to order (checkout)"""
    # Verify basket belongs to parent
    basket = db.query(Basket).filter(
        Basket.id == basket_id,
        Basket.parent_id == current_user.id
    ).first()
    
    if not basket:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Basket not found")
    
    if not basket.items:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Cannot checkout empty basket"
        )
    
    # Calculate total
    total = Decimal(0)
    for item in basket.items:
        unit_price = item.variant.trainer_price if item.variant.trainer_price else item.variant.admin_price
        total += unit_price * item.quantity
    
    # Create order
    order = Order(
        parent_id=current_user.id,
        child_id=basket.child_id,
        club_id=basket.club_id,
        status=OrderStatusEnum.PENDING,
        total_amount=total
    )
    db.add(order)
    db.flush()
    
    # Create order items from basket items
    for basket_item in basket.items:
        unit_price = basket_item.variant.trainer_price if basket_item.variant.trainer_price else basket_item.variant.admin_price
        order_item = OrderItem(
            order_id=order.id,
            product_id=basket_item.product_id,
            variant_id=basket_item.variant_id,
            quantity=basket_item.quantity,
            unit_price=unit_price,
            customization_text=basket_item.customization_text
        )
        db.add(order_item)
    
    # Delete basket and its items
    db.delete(basket)
    db.commit()
    
    return {"message": "Order created successfully", "order_id": order.id}


@router.delete("/{basket_id}", status_code=status.HTTP_204_NO_CONTENT)
def delete_basket(
    basket_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_parent)
):
    """Delete a basket"""
    basket = db.query(Basket).filter(
        Basket.id == basket_id,
        Basket.parent_id == current_user.id
    ).first()
    
    if not basket:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Basket not found")
    
    db.delete(basket)
    db.commit()
    
    return None
