from fastapi import APIRouter, Depends, HTTPException, status, Query
from sqlalchemy.orm import Session, joinedload, load_only
from sqlalchemy import func, and_
from typing import List, Optional
from datetime import datetime, date
from decimal import Decimal
from app.core import deps
from app.models import Order, OrderItem, Product, ProductVariant, User, Child, OrderStatusEnum
from app.schemas import order as order_schemas
from app.schemas.admin import DashboardStats

router = APIRouter()


@router.get("/stats", response_model=DashboardStats)
def get_dashboard_stats(
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Get dashboard statistics for admin - filtered by clubs they manage"""
    today = date.today()
    
    # Get club IDs that this admin manages
    managed_club_ids = [club.id for club in current_user.managed_clubs]
    club_filter = Order.club_id.in_(managed_club_ids) if managed_club_ids else True
    
    # Orders created today
    orders_today = db.query(func.count(Order.id)).filter(
        and_(func.date(Order.created_at) == today, club_filter)
    ).scalar()
    
    # Pending approvals (orders waiting for trainer approval)
    pending_approvals = db.query(func.count(Order.id)).filter(
        and_(Order.status == OrderStatusEnum.PENDING, club_filter)
    ).scalar()
    
    # Active work orders (not archived) for clubs admin manages
    from app.models.work_order import WorkOrder
    work_order_query = db.query(func.count(WorkOrder.id)).join(
        Order, WorkOrder.order_id == Order.id
    ).filter(WorkOrder.is_archived == False)
    
    if managed_club_ids:
        work_order_query = work_order_query.filter(Order.club_id.in_(managed_club_ids))
    
    active_work_orders = work_order_query.scalar()
    
    return DashboardStats(
        orders_today=orders_today or 0,
        pending_approvals=pending_approvals or 0,
        active_work_orders=active_work_orders or 0
    )


@router.get("/", response_model=List[order_schemas.OrderRead])
def list_orders(
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin),
    status_filter: Optional[OrderStatusEnum] = Query(None, alias="status"),
    club_id: Optional[int] = None,
    skip: int = 0,
    limit: int = 100
):
    """List all orders with optional filtering (admin only) - filtered by clubs admin manages"""
    query = db.query(Order).options(
        joinedload(Order.items).joinedload(OrderItem.variant),
        joinedload(Order.child)
    )
    
    # Admin can only see orders for clubs they manage
    managed_club_ids = [club.id for club in current_user.managed_clubs]
    if managed_club_ids:
        query = query.filter(Order.club_id.in_(managed_club_ids))
    elif club_id:
        query = query.filter(Order.club_id == club_id)
    
    if status_filter:
        query = query.filter(Order.status == status_filter)
    
    orders = query.order_by(Order.created_at.desc()).offset(skip).limit(limit).all()
    return orders


@router.get("/{order_id}", response_model=order_schemas.OrderRead)
def get_order(
    order_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Get a specific order with details (admin only)"""
    order = db.query(Order).options(
        joinedload(Order.items)
        .joinedload(OrderItem.product)
        .load_only(Product.name, Product.image_url),
        joinedload(Order.items)
        .joinedload(OrderItem.variant)
        .load_only(ProductVariant.size, ProductVariant.admin_price, ProductVariant.trainer_price)
    ).filter(Order.id == order_id).first()
    if not order:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Order not found")
    return order


@router.post("/", response_model=order_schemas.OrderRead, status_code=status.HTTP_201_CREATED)
def create_order(
    order_in: order_schemas.OrderCreate,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_user)
):
    """Create a new order (parents create orders for their children)"""
    # Verify child exists and belongs to parent
    child = db.query(Child).filter(Child.id == order_in.child_id).first()
    if not child:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Child not found")
    
    # For parents, verify they own the child
    if current_user.role.value == "PARENT" and child.parent_id != current_user.id:
        raise HTTPException(status_code=status.HTTP_403_FORBIDDEN, detail="Not authorized")
    
    # Calculate total and create order
    total_amount = Decimal(0)
    order_items_data = []
    
    for item in order_in.items:
        variant = db.query(ProductVariant).filter(ProductVariant.id == item.variant_id).first()
        if not variant:
            raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=f"Variant {item.variant_id} not found")
        
        unit_price = variant.subsidized_price if item.is_subsidized else variant.regular_price
        total_amount += unit_price * item.quantity
        
        order_items_data.append({
            "product_id": item.product_id,
            "variant_id": item.variant_id,
            "quantity": item.quantity,
            "unit_price": unit_price,
            "is_subsidized": item.is_subsidized,
            "customization_text": item.customization_text
        })
    
    # Create order
    order = Order(
        parent_id=current_user.id,
        child_id=order_in.child_id,
        club_id=child.club_id,
        total_amount=total_amount,
        notes=order_in.notes,
        status=OrderStatusEnum.PENDING
    )
    db.add(order)
    db.flush()
    
    # Create order items
    for item_data in order_items_data:
        order_item = OrderItem(**item_data, order_id=order.id)
        db.add(order_item)
    
    db.commit()
    db.refresh(order)
    return order


@router.patch("/{order_id}", response_model=order_schemas.OrderRead)
def update_order_status(
    order_id: int,
    order_update: order_schemas.OrderUpdate,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_user)
):
    """Update order status (trainers can approve/reject, admins can change to any status)"""
    order = db.query(Order).filter(Order.id == order_id).first()
    if not order:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Order not found")
    
    # Authorization checks
    if current_user.role.value == "TRAINER":
        # Trainers can only approve/reject orders from their club
        if order.club_id != current_user.club_id:
            raise HTTPException(status_code=status.HTTP_403_FORBIDDEN, detail="Not authorized")
        if order_update.status not in [OrderStatusEnum.APPROVED, OrderStatusEnum.REJECTED]:
            raise HTTPException(status_code=status.HTTP_403_FORBIDDEN, detail="Trainers can only approve or reject")
    elif current_user.role.value != "COMPANY_EMPLOYEE":
        raise HTTPException(status_code=status.HTTP_403_FORBIDDEN, detail="Not authorized")
    
    # Update order
    update_data = order_update.model_dump(exclude_unset=True)
    for field, value in update_data.items():
        setattr(order, field, value)
    
    db.commit()
    db.refresh(order)
    return order


# ===== TRAINER ENDPOINTS =====

@router.get("/trainer/pending", response_model=List[order_schemas.OrderListRead])
def list_trainer_pending_orders(
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_trainer),
    skip: int = 0,
    limit: int = 100
):
    """List all pending orders for trainer's club (trainer only)"""
    if not current_user.club_id:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Trainer not assigned to a club"
        )
    
    orders = db.query(Order).options(joinedload(Order.child)).filter(
        Order.club_id == current_user.club_id,
        Order.status == OrderStatusEnum.PENDING
    ).order_by(Order.created_at.desc()).offset(skip).limit(limit).all()
    
    return orders


@router.get("/trainer/all", response_model=List[order_schemas.OrderListRead])
def list_trainer_orders(
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_trainer),
    status_filter: Optional[OrderStatusEnum] = Query(None, alias="status"),
    skip: int = 0,
    limit: int = 100
):
    """List all orders for trainer's club (trainer only)"""
    if not current_user.club_id:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Trainer not assigned to a club"
        )
    
    query = db.query(Order).options(joinedload(Order.child)).filter(Order.club_id == current_user.club_id)
    
    if status_filter:
        query = query.filter(Order.status == status_filter)
    
    orders = query.order_by(Order.created_at.desc()).offset(skip).limit(limit).all()
    
    return orders


@router.get("/trainer/{order_id}", response_model=order_schemas.OrderRead)
def get_trainer_order(
    order_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_trainer)
):
    """Get specific order details (trainer only)"""
    order = db.query(Order).options(
        joinedload(Order.items)
        .joinedload(OrderItem.product)
        .load_only(Product.name, Product.image_url),
        joinedload(Order.items)
        .joinedload(OrderItem.variant)
        .load_only(ProductVariant.size, ProductVariant.admin_price, ProductVariant.trainer_price),
        joinedload(Order.child)
    ).filter(Order.id == order_id).first()
    
    if not order:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Order not found")
    
    # Verify order is from trainer's club
    if order.club_id != current_user.club_id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="You can only view orders from your club"
        )
    
    return order


@router.post("/trainer/{order_id}/approve", response_model=order_schemas.OrderRead)
def approve_order(
    order_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_trainer)
):
    """Approve an order and send it to admin (trainer only)"""
    order = db.query(Order).filter(Order.id == order_id).first()
    
    if not order:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Order not found")
    
    # Verify order is from trainer's club
    if order.club_id != current_user.club_id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="You can only approve orders from your club"
        )
    
    # Check order is in pending status
    if order.status != OrderStatusEnum.PENDING:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"Order is already {order.status.value}"
        )
    
    # Approve order
    order.status = OrderStatusEnum.APPROVED
    db.commit()
    db.refresh(order)
    
    return order


@router.post("/trainer/{order_id}/reject", response_model=order_schemas.OrderRead)
def reject_order(
    order_id: int,
    rejection_data: order_schemas.OrderRejection,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_trainer)
):
    """Reject an order with a reason (trainer only)"""
    order = db.query(Order).filter(Order.id == order_id).first()
    
    if not order:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Order not found")
    
    # Verify order is from trainer's club
    if order.club_id != current_user.club_id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="You can only reject orders from your club"
        )
    
    # Check order is in pending status
    if order.status != OrderStatusEnum.PENDING:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"Order is already {order.status.value}"
        )
    
    # Reject order
    order.status = OrderStatusEnum.REJECTED
    order.notes = f"Rejected by trainer: {rejection_data.reason}" + (f"\n\nOriginal notes: {order.notes}" if order.notes else "")
    db.commit()
    db.refresh(order)
    
    return order


# ===== PARENT ENDPOINTS =====

@router.get("/parent/my-orders", response_model=List[order_schemas.OrderListRead])
def list_parent_orders(
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_parent),
    skip: int = 0,
    limit: int = 100
):
    """List all orders for current parent"""
    orders = db.query(Order).filter(
        Order.parent_id == current_user.id
    ).order_by(Order.created_at.desc()).offset(skip).limit(limit).all()
    
    return orders


@router.get("/parent/{order_id}", response_model=order_schemas.OrderRead)
def get_parent_order(
    order_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_parent)
):
    """Get specific order details (parent only)"""
    order = db.query(Order).options(
        joinedload(Order.items).joinedload(OrderItem.product),
        joinedload(Order.items).joinedload(OrderItem.variant)
    ).filter(Order.id == order_id).first()
    
    if not order:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Order not found")
    
    # Verify order belongs to parent
    if order.parent_id != current_user.id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="You can only view your own orders"
        )
    
    return order


