from fastapi import APIRouter, Depends, HTTPException, status, Query
from sqlalchemy.orm import Session
from typing import List, Optional
from datetime import datetime
from app.core import deps
from app.models import WorkOrder, Order, User, Club, OrderStatusEnum
from app.schemas import work_order as work_order_schemas

router = APIRouter()


@router.get("/", response_model=List[work_order_schemas.WorkOrderRead])
def list_work_orders(
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin),
    archived: Optional[bool] = Query(None),
    skip: int = 0,
    limit: int = 100
):
    """List all work orders (admin only) - filtered by clubs admin manages"""
    query = db.query(WorkOrder)
    
    # Filter by clubs the admin manages
    managed_club_ids = [club.id for club in current_user.managed_clubs]
    if managed_club_ids:
        query = query.filter(WorkOrder.club_id.in_(managed_club_ids))
    
    if archived is not None:
        query = query.filter(WorkOrder.is_archived == archived)
    
    work_orders = query.order_by(WorkOrder.created_at.desc()).offset(skip).limit(limit).all()
    return work_orders


@router.get("/{work_order_id}", response_model=work_order_schemas.WorkOrderRead)
def get_work_order(
    work_order_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Get a specific work order (admin only)"""
    work_order = db.query(WorkOrder).filter(WorkOrder.id == work_order_id).first()
    if not work_order:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Work order not found")
    return work_order


@router.post("/", response_model=work_order_schemas.WorkOrderRead, status_code=status.HTTP_201_CREATED)
def create_work_order(
    work_order_in: work_order_schemas.WorkOrderCreate,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Create a work order from multiple approved orders (admin only)"""
    from datetime import datetime
    
    # Parse order IDs from comma-separated string
    order_ids_list = [int(oid.strip()) for oid in work_order_in.order_ids.split(',')]
    
    # Verify all orders exist and are approved
    orders = db.query(Order).filter(Order.id.in_(order_ids_list)).all()
    if len(orders) != len(order_ids_list):
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="One or more orders not found")
    
    for order in orders:
        if order.status != OrderStatusEnum.APPROVED:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail=f"Order {order.id} is not approved (status: {order.status})"
            )
    
    # Get club to extract club name
    club = db.query(Club).filter(Club.id == work_order_in.club_id).first()
    if not club:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Club not found")
    
    # Generate work order number: WO_CLB_YYYYMMDD_HHMM
    now = datetime.utcnow()
    date_str = now.strftime('%Y%m%d')  # YYYYMMDD
    time_str = now.strftime('%H%M')    # HHMM
    club_code = club.name[:3].upper()  # First 3 letters of club name
    work_order_number = f"WO_{club_code}_{date_str}_{time_str}"
    
    # Create work order with generated number
    work_order = WorkOrder(
        work_order_number=work_order_number,
        order_ids=work_order_in.order_ids,
        club_id=work_order_in.club_id,
        created_by_id=current_user.id,
        notes=work_order_in.notes
    )
    db.add(work_order)
    
    # Update all orders status to IN_PRODUCTION
    for order in orders:
        order.status = OrderStatusEnum.IN_PRODUCTION
    
    db.commit()
    db.refresh(work_order)
    return work_order


@router.patch("/{work_order_id}", response_model=work_order_schemas.WorkOrderRead)
def update_work_order(
    work_order_id: int,
    work_order_update: work_order_schemas.WorkOrderUpdate,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Update work order fields (admin only)"""
    work_order = db.query(WorkOrder).filter(WorkOrder.id == work_order_id).first()
    if not work_order:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Work order not found")
    
    if work_order_update.model is not None:
        work_order.model = work_order_update.model
    if work_order_update.due_date is not None:
        work_order.due_date = work_order_update.due_date
    if work_order_update.notes is not None:
        work_order.notes = work_order_update.notes
    
    db.commit()
    db.refresh(work_order)
    return work_order


@router.patch("/{work_order_id}/archive", response_model=work_order_schemas.WorkOrderRead)
def archive_work_order(
    work_order_id: int,
    archive_data: work_order_schemas.WorkOrderArchive,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Archive or unarchive a work order (admin only)"""
    work_order = db.query(WorkOrder).filter(WorkOrder.id == work_order_id).first()
    if not work_order:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Work order not found")
    
    work_order.is_archived = archive_data.is_archived
    if archive_data.is_archived:
        work_order.archived_at = datetime.utcnow()
        # Optionally update order status to completed
        if work_order.order:
            work_order.order.status = OrderStatusEnum.COMPLETED
    else:
        work_order.archived_at = None
    
    db.commit()
    db.refresh(work_order)
    return work_order


@router.delete("/{work_order_id}", status_code=status.HTTP_204_NO_CONTENT)
def delete_work_order(
    work_order_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Delete a work order (admin only)"""
    work_order = db.query(WorkOrder).filter(WorkOrder.id == work_order_id).first()
    if not work_order:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Work order not found")
    
    # Revert order status if needed
    if work_order.order and work_order.order.status == OrderStatusEnum.IN_PRODUCTION:
        work_order.order.status = OrderStatusEnum.APPROVED
    
    db.delete(work_order)
    db.commit()
    return None

