from sqlalchemy import Column, Integer, String, ForeignKey, Text, Boolean
from sqlalchemy.orm import relationship
from app.db.base import Base


class Product(Base):
    __tablename__ = "products"
    
    id = Column(Integer, primary_key=True, index=True)
    club_id = Column(Integer, ForeignKey("clubs.id"), nullable=False)
    name = Column(String, nullable=False)
    description = Column(Text, nullable=True)
    image_url = Column(Text, nullable=True)  # Changed to Text to support base64 images
    is_confirmed = Column(Boolean, nullable=False, default=False)  # Trainer confirms to make visible to parents
    custom_fields = Column(Text, nullable=True)  # JSON string for custom fields
    currency = Column(String, default='RSD', nullable=False)  # Currency: RSD or EUR
    has_customization = Column(Boolean, default=False, nullable=False)  # Whether product allows custom text/initials
    
    club = relationship("Club", back_populates="products")
    variants = relationship("ProductVariant", back_populates="product", cascade="all, delete-orphan")
    order_items = relationship("OrderItem", back_populates="product")


class ProductVariant(Base):
    __tablename__ = "product_variants"
    
    id = Column(Integer, primary_key=True, index=True)
    product_id = Column(Integer, ForeignKey("products.id"), nullable=False)
    size = Column(String, nullable=False)  # e.g., "S", "M", "L", "XL", "6", "38", etc.
    admin_price = Column(Integer, nullable=False)  # Base price in RSD (full numbers only)
    trainer_price = Column(Integer, nullable=True)  # Price set by trainer in RSD
    
    product = relationship("Product", back_populates="variants")
    order_items = relationship("OrderItem", back_populates="variant")
    
    @property
    def price(self):
        """Return trainer price if set, otherwise admin price"""
        return self.trainer_price if self.trainer_price is not None else self.admin_price
