from pydantic import BaseModel, field_validator
from typing import Optional, List


class ProductVariantBase(BaseModel):
    size: str
    admin_price: int  # RSD - full numbers only


class ProductVariantCreate(ProductVariantBase):
    pass


class ProductVariantUpdate(BaseModel):
    size: Optional[str] = None
    admin_price: Optional[int] = None


class ProductVariantSetPrice(BaseModel):
    """Schema for trainer to set their price"""
    trainer_price: int  # RSD - full numbers only
    
    @field_validator('trainer_price')
    @classmethod
    def validate_positive(cls, v):
        if v <= 0:
            raise ValueError('Price must be positive')
        return v


class ProductVariantRead(BaseModel):
    id: int
    product_id: int
    size: str
    admin_price: int
    trainer_price: Optional[int]
    price: int  # Computed property

    class Config:
        from_attributes = True


class ProductBase(BaseModel):
    name: str
    description: Optional[str] = None
    image_url: Optional[str] = None
    custom_fields: Optional[str] = None  # JSON string
    has_customization: bool = False


class ProductCreate(ProductBase):
    club_id: int
    variants: List[ProductVariantCreate]


class ProductUpdate(BaseModel):
    name: Optional[str] = None
    description: Optional[str] = None
    image_url: Optional[str] = None
    has_customization: Optional[bool] = None


class ProductConfirm(BaseModel):
    """Schema for trainer to confirm product visibility"""
    is_confirmed: bool


class ProductRead(ProductBase):
    id: int
    club_id: int
    is_confirmed: bool
    variants: List[ProductVariantRead]

    class Config:
        from_attributes = True


class ProductListRead(ProductBase):
    id: int
    club_id: int
    is_confirmed: bool

    class Config:
        from_attributes = True
