import secrets
from datetime import datetime, timedelta
from typing import Optional
from sqlalchemy.orm import Session
from passlib.context import CryptContext
from jose import jwt, JWTError
from app.models.user import User, RoleEnum
from app.models.invite import Invite
from app.models.club import Club
from app.models.child import Child
from app.core.config import settings
import smtplib
import ssl
import time
import logging
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart

pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")

ALGORITHM = "HS256"
ACCESS_TOKEN_EXPIRE_MINUTES = 15
REFRESH_TOKEN_EXPIRE_DAYS = 14
REFRESH_TOKEN_EXPIRE_SECONDS = REFRESH_TOKEN_EXPIRE_DAYS * 24 * 3600


def get_password_hash(password: str) -> str:
    return pwd_context.hash(password)


def verify_password(plain_password: str, hashed_password: str) -> bool:
    return pwd_context.verify(plain_password, hashed_password)


def authenticate_user(db: Session, email: str, password: str) -> Optional[User]:
    user = db.query(User).filter(User.email == email).first()
    if not user:
        return None
    if not verify_password(password, user.hashed_password):
        return None
    return user


def create_access_token_for_user(user: User) -> str:
    expire = datetime.utcnow() + timedelta(minutes=ACCESS_TOKEN_EXPIRE_MINUTES)
    to_encode = {"sub": str(user.id), "exp": expire, "role": user.role.value}
    return jwt.encode(to_encode, settings.SECRET_KEY, algorithm=ALGORITHM)


def create_refresh_token_for_user(user: User) -> str:
    expire = datetime.utcnow() + timedelta(days=REFRESH_TOKEN_EXPIRE_DAYS)
    to_encode = {"sub": str(user.id), "exp": expire, "type": "refresh"}
    return jwt.encode(to_encode, settings.SECRET_KEY, algorithm=ALGORITHM)


def verify_refresh_token(db: Session, token: str) -> Optional[User]:
    try:
        payload = jwt.decode(token, settings.SECRET_KEY, algorithms=[ALGORITHM])
        if payload.get("type") != "refresh":
            return None
        user_id = int(payload.get("sub"))
        user = db.query(User).get(user_id)
        return user
    except JWTError:
        return None


def create_password_reset_token(user: User) -> str:
    expire = datetime.utcnow() + timedelta(hours=1)
    to_encode = {"sub": str(user.id), "exp": expire, "type": "reset"}
    return jwt.encode(to_encode, settings.SECRET_KEY, algorithm=ALGORITHM)


def verify_password_reset_token(db: Session, token: str) -> Optional[User]:
    try:
        payload = jwt.decode(token, settings.SECRET_KEY, algorithms=[ALGORITHM])
        if payload.get("type") != "reset":
            return None
        user_id = int(payload.get("sub"))
        user = db.query(User).get(user_id)
        return user
    except JWTError:
        return None


def reset_password(db: Session, token: str, new_password: str) -> Optional[User]:
    user = verify_password_reset_token(db, token)
    if not user:
        return None
    user.hashed_password = get_password_hash(new_password)
    db.add(user)
    db.commit()
    return user


def create_trainer_invite_token(db: Session, admin_id: int, club_id: int | None = None, expiry_hours: int = 24) -> str:
    """Create a secure, time-limited invite token for trainer registration"""
    token = secrets.token_urlsafe(32)
    expires_at = datetime.utcnow() + timedelta(hours=expiry_hours)
    
    invite = Invite(
        token=token,
        role=RoleEnum.TRAINER.value,
        club_id=club_id,  # Bind to club if provided
        created_by=admin_id,
        expires_at=expires_at,
        used=False,
        revoked=False
    )
    db.add(invite)
    db.commit()
    return token


def list_admin_trainer_invites(db: Session, admin_id: int):
    """List all trainer invites created by an admin"""
    invites = db.query(Invite).filter(
        Invite.created_by == admin_id,
        Invite.role == RoleEnum.TRAINER.value
    ).order_by(Invite.created_at.desc()).all()
    
    result = []
    for inv in invites:
        status = "revoked" if inv.revoked else ("used" if inv.used else ("expired" if inv.expires_at < datetime.utcnow() else "active"))
        result.append({
            "token": inv.token,
            "created_at": inv.created_at.isoformat(),
            "expires_at": inv.expires_at.isoformat(),
            "used": inv.used,
            "used_at": inv.used_at.isoformat() if inv.used_at else None,
            "revoked": inv.revoked,
            "status": status
        })
    return result


def revoke_trainer_invite(db: Session, token: str, admin_id: int) -> bool:
    """Revoke a trainer invite token"""
    invite = db.query(Invite).filter(
        Invite.token == token,
        Invite.created_by == admin_id,
        Invite.role == RoleEnum.TRAINER.value,
        Invite.used == False
    ).first()
    
    if not invite:
        return False
    
    invite.revoked = True
    invite.revoked_at = datetime.utcnow()
    db.commit()
    return True


def validate_trainer_invite_token(db: Session, token: str):
    """Validate a trainer invite token (check expiry, usage, revocation)"""
    invite = db.query(Invite).filter(
        Invite.token == token,
        Invite.role == RoleEnum.TRAINER.value
    ).first()
    
    if not invite:
        raise ValueError("Invalid invite token")
    
    if invite.used:
        raise ValueError("Invite token has already been used")
    
    if invite.revoked:
        raise ValueError("Invite token has been revoked")
    
    current_time = datetime.utcnow()
    if invite.expires_at < current_time:
        # Log for debugging
        import logging
        logger = logging.getLogger(__name__)
        logger.info(f"Token expired. Expires at: {invite.expires_at}, Current time: {current_time}")
        raise ValueError("Invite token has expired")
    
    return invite
    
    return invite


def create_trainer_invite(db: Session, generated_by_id: int, club_name: str, city: Optional[str]) -> str:
    token = secrets.token_urlsafe(32)
    # create club and invite
    club = Club(name=club_name, city=city)
    db.add(club)
    db.flush()  # get id
    invite = Invite(token=token, role=RoleEnum.TRAINER.value, club_id=club.id, created_by=generated_by_id)
    db.add(invite)
    db.commit()
    return token


def create_parent_invite(db: Session, generated_by_id: int, club_id: int) -> str:
    """Create a secure, time-limited invite token for parent registration"""
    token = secrets.token_urlsafe(32)
    expires_at = datetime.utcnow() + timedelta(hours=24)
    
    invite = Invite(
        token=token,
        role=RoleEnum.PARENT.value,
        club_id=club_id,
        created_by=generated_by_id,
        expires_at=expires_at,
        used=False,
        revoked=False
    )
    db.add(invite)
    db.commit()
    return token


def list_trainer_parent_invites(db: Session, trainer_id: int):
    """List all parent invites created by a trainer"""
    invites = db.query(Invite).filter(
        Invite.created_by == trainer_id,
        Invite.role == RoleEnum.PARENT.value
    ).order_by(Invite.created_at.desc()).all()
    
    result = []
    for inv in invites:
        status = "revoked" if inv.revoked else ("used" if inv.used else ("expired" if inv.expires_at < datetime.utcnow() else "active"))
        result.append({
            "token": inv.token,
            "club_id": inv.club_id,
            "created_at": inv.created_at.isoformat(),
            "expires_at": inv.expires_at.isoformat(),
            "used": inv.used,
            "used_at": inv.used_at.isoformat() if inv.used_at else None,
            "revoked": inv.revoked,
            "status": status
        })
    return result


def revoke_parent_invite(db: Session, token: str, trainer_id: int) -> bool:
    """Revoke a parent invite token"""
    invite = db.query(Invite).filter(
        Invite.token == token,
        Invite.created_by == trainer_id,
        Invite.role == RoleEnum.PARENT.value,
        Invite.used == False
    ).first()
    
    if not invite:
        return False
    
    invite.revoked = True
    invite.revoked_at = datetime.utcnow()
    db.commit()
    return True


def validate_parent_invite_token(db: Session, token: str):
    """Validate a parent invite token (check expiry, usage, revocation)"""
    invite = db.query(Invite).filter(
        Invite.token == token,
        Invite.role == RoleEnum.PARENT.value
    ).first()
    
    if not invite:
        raise ValueError("Invalid invite token")
    
    if invite.used:
        raise ValueError("Invite token has already been used")
    
    if invite.revoked:
        raise ValueError("Invite token has been revoked")
    
    if invite.expires_at < datetime.utcnow():
        raise ValueError("Invite token has expired")
    
    return invite



def register_trainer(db: Session, payload):
    # New secure token system (recommended)
    if payload.token:
        invite = validate_trainer_invite_token(db, payload.token)
        admin = db.query(User).filter(User.id == invite.created_by).first()
        
        if not admin:
            raise ValueError("Invite creator not found")
        
        # Check if invite is already bound to a club
        if invite.club_id:
            # Use the existing club from the invite
            club = db.query(Club).filter(Club.id == invite.club_id).first()
            if not club:
                raise ValueError("Club associated with invite not found")
        else:
            # Create new club for the trainer
            club = Club(name=payload.club_name, city=payload.city, admin_id=admin.id)
            db.add(club)
            db.flush()
        
        hashed = get_password_hash(payload.password)
        user = User(
            email=payload.email,
            hashed_password=hashed,
            first_name=payload.first_name,
            last_name=payload.last_name,
            role=RoleEnum.TRAINER,
            club_id=club.id
        )
        db.add(user)
        
        # Mark invite as used
        invite.used = True
        invite.used_at = datetime.utcnow()
        db.commit()
        return user
    
    # Legacy: Direct admin_id (deprecated for security)
    if payload.admin_id:
        admin = db.query(User).filter(User.id == payload.admin_id, User.role == RoleEnum.COMPANY_EMPLOYEE).first()
        if not admin:
            raise ValueError("Invalid admin ID")
        
        club = Club(name=payload.club_name, city=payload.city, admin_id=admin.id)
        db.add(club)
        db.flush()
        
        hashed = get_password_hash(payload.password)
        user = User(
            email=payload.email, 
            hashed_password=hashed, 
            first_name=payload.first_name, 
            last_name=payload.last_name, 
            role=RoleEnum.TRAINER, 
            club_id=club.id
        )
        db.add(user)
        db.commit()
        return user
    
    # Legacy invite token system
    if payload.invite_token:
        invite = db.query(Invite).filter(Invite.token == payload.invite_token, Invite.used == False).first()
        if not invite or invite.role != RoleEnum.TRAINER.value:
            raise ValueError("Invalid or used invite")
        club = db.query(Club).get(invite.club_id)
        if not club:
            club = Club(name=payload.club_name, city=payload.city)
            db.add(club)
            db.flush()
        hashed = get_password_hash(payload.password)
        user = User(email=payload.email, hashed_password=hashed, first_name=payload.first_name, last_name=payload.last_name, role=RoleEnum.TRAINER, club_id=club.id)
        db.add(user)
        invite.used = True
        db.commit()
        return user
    
    raise ValueError("Token, admin_id, or invite_token is required")


def register_parent(db: Session, payload):
    """Register a parent using invite token from trainer"""
    # Validate the parent invite token
    invite = validate_parent_invite_token(db, payload.invite_token)
    
    # Get the club from the invite
    club = db.query(Club).get(invite.club_id)
    if not club:
        raise ValueError("Invite references missing club")
    
    # Check if email already exists
    existing_user = db.query(User).filter(User.email == payload.email).first()
    if existing_user:
        raise ValueError("Email already registered")
    
    # Create parent user
    hashed = get_password_hash(payload.password)
    user = User(
        email=payload.email,
        hashed_password=hashed,
        first_name=payload.first_name,
        last_name=payload.last_name,
        role=RoleEnum.PARENT,
        club_id=club.id
    )
    db.add(user)
    db.flush()
    
    # Mark invite as used
    invite.used = True
    invite.used_at = datetime.utcnow()
    invite.used_by = user.id
    
    db.commit()
    db.refresh(user)
    return user

# SEND PASSWORD RESET EMAIL FUNCTIONALITY

logger = logging.getLogger(__name__)

def send_email_smtp(
    host: str,
    port: int,
    username: str,
    password: str,
    sender: str,
    recipients: list[str],
    subject: str,
    text_body: str,
    html_body: str | None = None,
    use_tls: bool = True,
) -> None:
    """Send a simple multipart email via SMTP."""
    msg = MIMEMultipart("alternative")
    msg["Subject"] = subject
    msg["From"] = sender
    msg["To"] = ", ".join(recipients)

    part1 = MIMEText(text_body, "plain")
    msg.attach(part1)
    if html_body:
        part2 = MIMEText(html_body, "html")
        msg.attach(part2)

    context = ssl.create_default_context()
    if use_tls:
        with smtplib.SMTP(host, port, timeout=10) as server:
            server.starttls(context=context)
            if username and password:
                server.login(username, password)
            server.sendmail(sender, recipients, msg.as_string())
    else:
        with smtplib.SMTP_SSL(host, port, context=context, timeout=10) as server:
            if username and password:
                server.login(username, password)
            server.sendmail(sender, recipients, msg.as_string())


def send_password_reset_email(db: Session, email: str) -> None:
    """
    Create a password reset token and send it by email.
    Always returns quickly and does not reveal whether the email exists.
    """
    from app.models.user import User as UserModel

    user = db.query(UserModel).filter(UserModel.email == email).first()
    if not user:
        # don't reveal existence — add small delay to make timing similar
        time.sleep(0.35)
        logger.info("Password reset requested for unknown email: %s", email)
        return

    token = create_password_reset_token(user)
    frontend = getattr(settings, "FRONTEND_URL", "").rstrip("/") or "http://localhost:5173"
    reset_link = f"{frontend}/reset-password?token={token}"

    subject = "Password reset instructions"
    text = f"""Hello {user.first_name or ''},

We received a request to reset your password. Click the link below to reset it:

{reset_link}

If you didn't request a password reset, you can ignore this message.
"""
    html = f"""
    <p>Hello {user.first_name or ''},</p>
    <p>We received a request to reset your password. Click the link below to reset it:</p>
    <p><a href="{reset_link}">Reset password</a></p>
    <p>If you didn't request a password reset, you can ignore this message.</p>
    """

    # SMTP configuration from settings (set these in backend/.env)
    smtp_host = getattr(settings, "SMTP_HOST", None)
    smtp_port = int(getattr(settings, "SMTP_PORT", 587))
    smtp_user = getattr(settings, "SMTP_USER", None)
    smtp_pass = getattr(settings, "SMTP_PASSWORD", None)
    smtp_from = getattr(settings, "SMTP_FROM", smtp_user or f"no-reply@{smtp_host or 'localhost'}")
    smtp_use_tls = bool(getattr(settings, "SMTP_USE_TLS", True))

    if not smtp_host:
        # dev fallback: log token (do NOT return token to client in production)
        logger.warning("SMTP not configured; password reset token for %s: %s", email, token)
        return

    try:
        send_email_smtp(
            host=smtp_host,
            port=smtp_port,
            username=smtp_user,
            password=smtp_pass,
            sender=smtp_from,
            recipients=[user.email],
            subject=subject,
            text_body=text,
            html_body=html,
            use_tls=smtp_use_tls,
        )
        logger.info("Sent password reset email to user_id=%s email=%s", user.id, user.email)
    except Exception:
        logger.exception("Failed to send password reset email to %s", email)
