# ...existing code...
import os
from app.db.session import SessionLocal, engine
from app.db.base import Base

# ensure all model modules are imported so tables are registered on Base.metadata
# import specific modules to avoid circular imports at runtime
from app.models import user, club, invite, child, product, order, work_order  # noqa: F401

# prefer project's password hasher; fallback to simple passlib if unavailable
try:
    from app.services.auth import get_password_hash
except Exception:
    from passlib.context import CryptContext
    _pwd_ctx = CryptContext(schemes=["bcrypt"], deprecated="auto")
    def get_password_hash(pw: str) -> str:
        return _pwd_ctx.hash(pw)

from app.models.user import User, RoleEnum


def run():
    """Create tables and seed one COMPANY_EMPLOYEE user if missing."""
    # now that models are imported, create_all will see all tables including clubs
    Base.metadata.create_all(bind=engine)
    db = SessionLocal()
    try:
        admin_email = os.getenv("SEED_ADMIN_EMAIL", "marija.jolovic.kg@gmail.com")
        admin_password = os.getenv("SEED_ADMIN_PASSWORD", "password123")

        existing = db.query(User).filter(User.email == admin_email).first()
        if existing:
            print(f"Admin already exists: {admin_email} (id={existing.id})")
            return

        user = User(
            email=admin_email,
            hashed_password=get_password_hash(admin_password),
            first_name="Admin",          
            last_name="User",
            role=RoleEnum.COMPANY_EMPLOYEE,
            is_active=True,
        )
        db.add(user)
        db.commit()
        db.refresh(user)
        print(f"Created admin: {admin_email} / {admin_password}")
    finally:
        db.close()


if __name__ == "__main__":
    run()
# ...existing code...